/*============================================================================
	UnSocket.h: Common interface for WinSock and BSD sockets.

	Revision history:
		* Created by Mike Danylchuk
============================================================================*/

/*-----------------------------------------------------------------------------
	Definitions.
-----------------------------------------------------------------------------*/

#if __WINSOCK__
	typedef INT					SOCKLEN;
	#define GCC_OPT_INT_CAST
	#define MSG_NOSIGNAL		0
	#define UDP_ERR_PORT_UNREACH WSAECONNRESET
	#define WSAGetLastHostError()	WSAGetLastError()

	#ifdef UNICODE
		#define IpDrvGetHostName GetHostNameW
	#else
		#define IpDrvGetHostName gethostname
	#endif
#endif

// Provide WinSock definitions for BSD sockets.
#if __BSD_SOCKETS__
	#if __BSD_SOCKETS_TRUE_BSD__
		typedef int socklen_t;
	#endif

	typedef int			SOCKET;
	typedef struct hostent		HOSTENT;
	typedef in_addr			IN_ADDR;
	typedef struct sockaddr		SOCKADDR;
	typedef struct sockaddr_in	SOCKADDR_IN;
	typedef struct linger		LINGER;
	typedef struct timeval		TIMEVAL;
	typedef TCHAR*			LPSTR;
	typedef socklen_t		SOCKLEN;

	#define INVALID_SOCKET		-1
	#define SOCKET_ERROR		-1

	#define UDP_ERR_PORT_UNREACH ECONNREFUSED

	#define WSAEWOULDBLOCK		EAGAIN
	#define WSAEINPROGRESS		EINPROGRESS

	#define WSAENOTSOCK		ENOTSOCK
	#define WSATRY_AGAIN		TRY_AGAIN
	#define WSAHOST_NOT_FOUND	HOST_NOT_FOUND
	#define WSANO_DATA		NO_ADDRESS
	#define LPSOCKADDR		sockaddr*
	#define WSAECONNRESET		ECONNRESET

	#define closesocket		close
	#define ioctlsocket		ioctl
	#define WSAGetLastError()	errno

	#if __BSD_SOCKETS_TRUE_BSD__
		#define WSAGetLastHostError()	errno
	#else
		#define WSAGetLastHostError()	h_errno
	#endif

	#define GCC_OPT_INT_CAST	(DWORD*)
	#define InetPton inet_pton
	#define InetNtop inet_ntop
	#define IpDrvGetHostName gethostname
#endif

// IP address macros.
#if __WINSOCK__
	#define IP(sin_addr,n) sin_addr.S_un.S_un_b.s_b##n
#elif __BSD_SOCKETS__
	#define IP(sin_addr,n) ((BYTE*)&sin_addr.s_addr)[n-1]
#endif

/*-----------------------------------------------------------------------------
	Definitions.
-----------------------------------------------------------------------------*/

// An IP address (host byte order).
struct IPDRV_API FIpAddr
{
	INT Family;
	union {
		in_addr Addr; // network byte order
		in6_addr Addr6; // network byte order
	};
	DWORD Port; // Host byte order

	// tors
	FIpAddr() : Family(AF_UNSPEC), Port() {}
	FIpAddr( const TCHAR* IPString, INT InPort )
	{
		Port = InPort;

		int result = InetPton(AF_INET6, IPString, &Addr6);
		if (result == 1) {
			Family = AF_INET6;
			return;
		}
		result = InetPton(AF_INET, IPString, &Addr);
		if (result == 1) {
			Family = AF_INET;
			return;
		}
		
		Family = AF_UNSPEC;
	}
	FIpAddr(sockaddr* SockAddr);
private:
	FIpAddr(sockaddr_in SockAddr);
	FIpAddr(sockaddr_in6 SockAddr);
public:

	// equality
	UBOOL operator==(FIpAddr& Other)
	{
		if (Family != Other.Family) return false;
		if (Family == AF_INET)
			return memcmp(&Addr, &Other.Addr, sizeof(Addr)) == 0 && Port == Other.Port;
		else
			return memcmp(&Addr6, &Other.Addr6, sizeof(Addr6)) == 0 && Port == Other.Port;
	}

	// helpers
	FString GetString( UBOOL ShowPort );
	UBOOL GetSockAddr(sockaddr_storage* Storage, socklen_t* StorageLength = nullptr);
	UBOOL IsBroadcast() {
		return Family == AF_INET && Addr.s_addr == INADDR_BROADCAST;
	}

	// serialization
    friend IPDRV_API FArchive& operator<<( FArchive& Ar, FIpAddr& IP )
    {
		if (Ar.Ver() < 129) {
			IP.Family = AF_INET;
			Ar << IP.Addr.s_addr;
			IP.Addr.s_addr = htonl(IP.Addr.s_addr);
		} else {
			Ar << IP.Family;
			if (IP.Family == AF_INET)
				Ar << IP.Addr.s_addr;
			else for (auto&& w : IP.Addr6.s6_words)
				Ar << w;
		}
		Ar << IP.Port;
        return Ar;
    }
};

// SocketData.
struct IPDRV_API FSocketData
{
	sockaddr_storage Addr;	// Network byte order.
	socklen_t AddrLen;
	INT Port;			// Host byte order.
	SOCKET Socket;

	void UpdateFromSocket()
	{
		SOCKLEN size = sizeof(Addr);
		getsockname ( Socket, (SOCKADDR*)(&Addr), &AddrLen);
		Port = ntohs(((sockaddr_in*)&Addr)->sin_port);
	}
	FString GetString( UBOOL ShowPort );
};

/*----------------------------------------------------------------------------
	Functions.
----------------------------------------------------------------------------*/

UBOOL InitSockets( FString& Error );
UBOOL SetNonBlocking( INT Socket );
UBOOL SetSocketReuseAddr( INT Socket, UBOOL ReUse=1 );
UBOOL SetSocketLinger( INT Socket );
UBOOL SetSocketRecvErr( INT Socket );
const TCHAR* SocketError( INT Code=-1 );
const TCHAR* SocketHostError( INT Code=-1 );
UBOOL IpMatches( sockaddr* A, sockaddr* B );
UBOOL IpAddressMatches( sockaddr* A, sockaddr* B );
UBOOL IpAddressRangeMatches( sockaddr* A, sockaddr* B );
void IpGetBytes( in_addr Addr, BYTE& Ip1, BYTE& Ip2, BYTE& Ip3, BYTE& Ip4 );
void IpSetBytes( in_addr& Addr, BYTE Ip1, BYTE Ip2, BYTE Ip3, BYTE Ip4 );
void IpGetInt( in_addr Addr, DWORD& Ip );
void IpSetInt( in_addr& Addr, DWORD Ip );
FString IpString( in_addr Addr, INT Port=0 );
FString IpString(sockaddr* Addr, socklen_t AddrLen, UBOOL ShowPort = false);

/*----------------------------------------------------------------------------
	The End.
----------------------------------------------------------------------------*/

